<?php
require_once 'config.php';
require_once 'database.php';

class TelegramBot {
    private $bot_token;
    private $db;
    private $api_url;
    
    public function __construct() {
        $this->bot_token = BOT_TOKEN;
        $this->db = new Database();
        $this->api_url = "https://api.telegram.org/bot" . $this->bot_token;
    }
    
    // ارسال پیام
    public function sendMessage($chat_id, $text, $reply_markup = null) {
        $data = [
            'chat_id' => $chat_id,
            'text' => $text,
            'parse_mode' => 'HTML'
        ];
        
        if ($reply_markup) {
            $data['reply_markup'] = json_encode($reply_markup);
        }
        
        return $this->makeRequest('sendMessage', $data);
    }
    
    // ویرایش پیام
    public function editMessageText($chat_id, $message_id, $text, $reply_markup = null) {
        $data = [
            'chat_id' => $chat_id,
            'message_id' => $message_id,
            'text' => $text,
            'parse_mode' => 'HTML'
        ];
        
        if ($reply_markup) {
            $data['reply_markup'] = json_encode($reply_markup);
        }
        
        return $this->makeRequest('editMessageText', $data);
    }
    
    // حذف پیام
    public function deleteMessage($chat_id, $message_id) {
        $data = [
            'chat_id' => $chat_id,
            'message_id' => $message_id
        ];
        
        return $this->makeRequest('deleteMessage', $data);
    }
    
    // ارسال درخواست به API تلگرام
    private function makeRequest($method, $data) {
        $url = $this->api_url . '/' . $method;
        
        $options = [
            'http' => [
                'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                'method' => 'POST',
                'content' => http_build_query($data)
            ]
        ];
        
        $context = stream_context_create($options);
        $result = file_get_contents($url, false, $context);
        
        return json_decode($result, true);
    }
    
    // پردازش webhook
    public function processWebhook() {
        $input = file_get_contents('php://input');
        $update = json_decode($input, true);
        
        if (!$update) {
            return false;
        }
        
        $this->db->log(null, 'webhook_received', $input);
        
        if (isset($update['message'])) {
            $this->handleMessage($update['message']);
        } elseif (isset($update['callback_query'])) {
            $this->handleCallbackQuery($update['callback_query']);
        }
        
        return true;
    }
    
    // پردازش پیام‌ها
    private function handleMessage($message) {
        $chat_id = $message['chat']['id'];
        $user_id = $message['from']['id'];
        $text = $message['text'] ?? '';
        
        // دریافت یا ایجاد کاربر
        $user = $this->db->getUser($user_id, $message['from']);
        
        if (!$user) {
            $this->sendMessage($chat_id, "❌ خطا در ایجاد حساب کاربری");
            return;
        }
        
        // ثبت لاگ
        $this->db->log($user['id'], 'message_received', $text);
        
        // پردازش دستورات
        if ($text === '/start') {
            $this->showMainMenu($chat_id, $user);
        } elseif ($text === '/menu') {
            $this->showMainMenu($chat_id, $user);
        } else {
            $this->handleTextMessage($chat_id, $user, $text);
        }
    }
    
    // پردازش callback query
    private function handleCallbackQuery($callback_query) {
        $chat_id = $callback_query['message']['chat']['id'];
        $user_id = $callback_query['from']['id'];
        $data = $callback_query['data'];
        $message_id = $callback_query['message']['message_id'];
        
        $user = $this->db->getUser($user_id);
        
        if (!$user) {
            $this->answerCallbackQuery($callback_query['id'], "❌ خطا در شناسایی کاربر");
            return;
        }
        
        $this->db->log($user['id'], 'callback_received', $data);
        
        // پردازش callback data
        $this->handleCallbackData($chat_id, $user, $data, $message_id);
        
        // پاسخ به callback query
        $this->answerCallbackQuery($callback_query['id']);
    }
    
    // پاسخ به callback query
    private function answerCallbackQuery($callback_query_id, $text = null) {
        $data = ['callback_query_id' => $callback_query_id];
        if ($text) {
            $data['text'] = $text;
        }
        
        $this->makeRequest('answerCallbackQuery', $data);
    }
    
    // نمایش منوی اصلی
    public function showMainMenu($chat_id, $user) {
        $text = "🤖 <b>خوش آمدید به ربات سفارش ربات!</b>\n\n";
        $text .= "👤 کاربر: " . ($user['first_name'] ?? 'نامشخص') . "\n";
        $text .= "💰 موجودی: " . number_format($user['balance']) . " " . CURRENCY . "\n\n";
        $text .= "لطفاً یکی از گزینه‌های زیر را انتخاب کنید:";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '👤 حساب کاربری', 'callback_data' => 'account'],
                    ['text' => '💳 افزایش موجودی', 'callback_data' => 'deposit']
                ],
                [
                    ['text' => '🤖 ثبت سفارش ربات', 'callback_data' => 'order_bot'],
                    ['text' => '📝 درخواست مشاوره', 'callback_data' => 'consultation']
                ],
                [
                    ['text' => '📜 قوانین و مقررات', 'callback_data' => 'rules']
                ]
            ]
        ];
        
        $this->sendMessage($chat_id, $text, $keyboard);
    }
    
    // پردازش callback data
    private function handleCallbackData($chat_id, $user, $data, $message_id) {
        switch ($data) {
            case 'account':
                $this->showAccount($chat_id, $user, $message_id);
                break;
            case 'deposit':
                $this->showDeposit($chat_id, $user, $message_id);
                break;
            case 'order_bot':
                $this->showBotList($chat_id, $user, $message_id);
                break;
            case 'consultation':
                $this->showConsultation($chat_id, $user, $message_id);
                break;
            case 'rules':
                $this->showRules($chat_id, $user, $message_id);
                break;
            case 'back_to_menu':
                $this->showMainMenu($chat_id, $user);
                break;
            default:
                if (strpos($data, 'bot_') === 0) {
                    $bot_type = substr($data, 4);
                    $this->showBotDetails($chat_id, $user, $bot_type, $message_id);
                } elseif (strpos($data, 'order_') === 0) {
                    $bot_type = substr($data, 6);
                    $this->processBotOrder($chat_id, $user, $bot_type, $message_id);
                }
                break;
        }
    }
    
    // نمایش حساب کاربری
    private function showAccount($chat_id, $user, $message_id) {
        $text = "👤 <b>حساب کاربری شما</b>\n\n";
        $text .= "🆔 شناسه: " . $user['telegram_id'] . "\n";
        $text .= "👤 نام: " . ($user['first_name'] ?? 'نامشخص') . "\n";
        $text .= "💰 موجودی: " . number_format($user['balance']) . " " . CURRENCY . "\n";
        $text .= "📅 تاریخ عضویت: " . date('Y/m/d', strtotime($user['created_at'])) . "\n\n";
        
        // دریافت آخرین تراکنش‌ها
        $transactions = $this->db->getUserTransactions($user['id'], 5);
        if ($transactions) {
            $text .= "📊 <b>آخرین تراکنش‌ها:</b>\n";
            foreach ($transactions as $transaction) {
                $type_icon = $transaction['type'] === 'deposit' ? '➕' : '➖';
                $text .= $type_icon . " " . number_format($transaction['amount']) . " " . CURRENCY . " - ";
                $text .= date('Y/m/d H:i', strtotime($transaction['created_at'])) . "\n";
            }
        }
        
        $keyboard = [
            'inline_keyboard' => [
                [['text' => '🔙 بازگشت به منو', 'callback_data' => 'back_to_menu']]
            ]
        ];
        
        $this->editMessageText($chat_id, $message_id, $text, $keyboard);
    }
    
    // نمایش صفحه افزایش موجودی
    private function showDeposit($chat_id, $user, $message_id) {
        $text = "💳 <b>افزایش موجودی</b>\n\n";
        $text .= "💰 موجودی فعلی: " . number_format($user['balance']) . " " . CURRENCY . "\n\n";
        $text .= "برای افزایش موجودی، روی دکمه زیر کلیک کنید:\n";
        $text .= "🔗 <a href='" . PAYMENT_PAGE_URL . "?user_id=" . $user['id'] . "'>صفحه پرداخت</a>";
        
        $keyboard = [
            'inline_keyboard' => [
                [['text' => '🔙 بازگشت به منو', 'callback_data' => 'back_to_menu']]
            ]
        ];
        
        $this->editMessageText($chat_id, $message_id, $text, $keyboard);
    }
    
    // نمایش لیست ربات‌ها
    private function showBotList($chat_id, $user, $message_id) {
        global $available_bots;
        
        $text = "🤖 <b>انتخاب ربات</b>\n\n";
        $text .= "لطفاً یکی از ربات‌های زیر را انتخاب کنید:\n\n";
        
        $keyboard = ['inline_keyboard' => []];
        
        foreach ($available_bots as $type => $bot) {
            $text .= "🤖 <b>" . $bot['name'] . "</b>\n";
            $text .= "💰 قیمت: " . number_format($bot['price']) . " " . CURRENCY . "\n";
            $text .= "📝 " . $bot['description'] . "\n\n";
            
            $keyboard['inline_keyboard'][] = [
                ['text' => "انتخاب " . $bot['name'], 'callback_data' => 'bot_' . $type]
            ];
        }
        
        $keyboard['inline_keyboard'][] = [
            ['text' => '🔙 بازگشت به منو', 'callback_data' => 'back_to_menu']
        ];
        
        $this->editMessageText($chat_id, $message_id, $text, $keyboard);
    }
    
    // نمایش جزئیات ربات
    private function showBotDetails($chat_id, $user, $bot_type, $message_id) {
        global $available_bots;
        
        if (!isset($available_bots[$bot_type])) {
            $this->editMessageText($chat_id, $message_id, "❌ ربات مورد نظر یافت نشد");
            return;
        }
        
        $bot = $available_bots[$bot_type];
        
        $text = "🤖 <b>" . $bot['name'] . "</b>\n\n";
        $text .= "💰 <b>قیمت:</b> " . number_format($bot['price']) . " " . CURRENCY . "\n\n";
        $text .= "📝 <b>توضیحات:</b>\n" . $bot['description'] . "\n\n";
        $text .= "🔧 <b>امکانات:</b>\n";
        
        // امکانات خاص هر ربات
        switch ($bot_type) {
            case 'downloader':
                $text .= "• دانلود فایل از لینک‌های مختلف\n";
                $text .= "• پشتیبانی از فرمت‌های مختلف\n";
                $text .= "• مدیریت صف دانلود\n";
                $text .= "• گزارش‌گیری از دانلودها\n";
                break;
            case 'shop':
                $text .= "• مدیریت محصولات\n";
                $text .= "• سبد خرید\n";
                $text .= "• سیستم پرداخت\n";
                $text .= "• پنل مدیریت\n";
                break;
            case 'group_manager':
                $text .= "• مدیریت اعضا\n";
                $text .= "• فیلتر پیام‌ها\n";
                $text .= "• سیستم ادمین\n";
                $text .= "• آمار گروه\n";
                break;
            case 'ticketing':
                $text .= "• سیستم تیکت\n";
                $text .= "• پشتیبانی چندسطحی\n";
                $text .= "• پیگیری وضعیت\n";
                $text .= "• گزارش‌گیری\n";
                break;
            case 'quiz':
                $text .= "• ساخت آزمون\n";
                $text .= "• مدیریت سوالات\n";
                $text .= "• سیستم نمره‌دهی\n";
                $text .= "• گزارش نتایج\n";
                break;
        }
        
        $text .= "\n💰 موجودی شما: " . number_format($user['balance']) . " " . CURRENCY . "\n";
        
        if ($user['balance'] >= $bot['price']) {
            $text .= "\n✅ موجودی شما کافی است!";
        } else {
            $text .= "\n❌ موجودی شما کافی نیست!";
        }
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '✅ سفارش این ربات', 'callback_data' => 'order_' . $bot_type],
                    ['text' => '💳 افزایش موجودی', 'callback_data' => 'deposit']
                ],
                [['text' => '🔙 بازگشت به لیست', 'callback_data' => 'order_bot']]
            ]
        ];
        
        $this->editMessageText($chat_id, $message_id, $text, $keyboard);
    }
    
    // پردازش سفارش ربات
    private function processBotOrder($chat_id, $user, $bot_type, $message_id) {
        global $available_bots;
        
        if (!isset($available_bots[$bot_type])) {
            $this->editMessageText($chat_id, $message_id, "❌ ربات مورد نظر یافت نشد");
            return;
        }
        
        $bot = $available_bots[$bot_type];
        
        if ($user['balance'] < $bot['price']) {
            $text = "❌ <b>موجودی ناکافی!</b>\n\n";
            $text .= "💰 موجودی شما: " . number_format($user['balance']) . " " . CURRENCY . "\n";
            $text .= "💰 قیمت ربات: " . number_format($bot['price']) . " " . CURRENCY . "\n";
            $text .= "💸 کمبود: " . number_format($bot['price'] - $user['balance']) . " " . CURRENCY . "\n\n";
            $text .= "لطفاً ابتدا موجودی خود را افزایش دهید.";
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '💳 افزایش موجودی', 'callback_data' => 'deposit']],
                    [['text' => '🔙 بازگشت', 'callback_data' => 'bot_' . $bot_type]]
                ]
            ];
            
            $this->editMessageText($chat_id, $message_id, $text, $keyboard);
            return;
        }
        
        // درخواست توکن ربات
        $text = "🤖 <b>سفارش " . $bot['name'] . "</b>\n\n";
        $text .= "✅ موجودی شما کافی است!\n\n";
        $text .= "لطفاً توکن ربات خود را ارسال کنید:\n";
        $text .= "🔗 <a href='https://t.me/BotFather'>@BotFather</a> → /newbot → توکن را کپی کنید\n\n";
        $text .= "⚠️ توجه: توکن را به صورت کامل و صحیح ارسال کنید.";
        
        $keyboard = [
            'inline_keyboard' => [
                [['text' => '❌ لغو سفارش', 'callback_data' => 'bot_' . $bot_type]]
            ]
        ];
        
        $this->editMessageText($chat_id, $message_id, $text, $keyboard);
        
        // ذخیره وضعیت درخواست توکن
        $_SESSION['waiting_for_token'] = $bot_type;
    }
    
    // نمایش صفحه مشاوره
    private function showConsultation($chat_id, $user, $message_id) {
        $text = "📝 <b>درخواست مشاوره</b>\n\n";
        $text .= "برای دریافت مشاوره، لطفاً اطلاعات زیر را ارسال کنید:\n\n";
        $text .= "1️⃣ موضوع مشاوره\n";
        $text .= "2️⃣ توضیحات کامل نیاز شما\n";
        $text .= "3️⃣ آیدی تلگرام یا شماره تماس\n\n";
        $text .= "📞 <b>مثال:</b>\n";
        $text .= "موضوع: طراحی ربات فروشگاه\n";
        $text .= "توضیحات: نیاز به ربات فروشگاه با قابلیت پرداخت آنلاین دارم\n";
        $text .= "تماس: @username یا 09123456789";
        
        $keyboard = [
            'inline_keyboard' => [
                [['text' => '🔙 بازگشت به منو', 'callback_data' => 'back_to_menu']]
            ]
        ];
        
        $this->editMessageText($chat_id, $message_id, $text, $keyboard);
        
        // ذخیره وضعیت درخواست مشاوره
        $_SESSION['waiting_for_consultation'] = true;
    }
    
    // نمایش قوانین
    private function showRules($chat_id, $user, $message_id) {
        $text = "📜 <b>قوانین و مقررات</b>\n\n";
        $text .= "🔴 <b>ممنوعیت‌ها:</b>\n";
        $text .= "• محتوای غیرمجاز (قمار، پورن، سیاسی)\n";
        $text .= "• محتوای فیلترشده و غیرقانونی\n";
        $text .= "• هرگونه محتوای خلاف قوانین جمهوری اسلامی ایران\n\n";
        $text .= "⚖️ <b>مسئولیت‌ها:</b>\n";
        $text .= "• مسئولیت محتوا بر عهده کاربر است\n";
        $text .= "• کاربر موظف به رعایت قوانین است\n";
        $text .= "• محرمانگی اطلاعات کاربران محفوظ است\n\n";
        $text .= "💰 <b>مالی:</b>\n";
        $text .= "• هزینه‌ها غیرقابل بازگشت مگر در صورت خطا از سمت ما\n";
        $text .= "• قیمت‌ها به تومان و نهایی است\n";
        $text .= "• پرداخت‌ها از طریق زرین‌پال انجام می‌شود\n\n";
        $text .= "✅ <b>پذیرش:</b>\n";
        $text .= "استفاده از ربات به معنای پذیرش کامل این قوانین است.";
        
        $keyboard = [
            'inline_keyboard' => [
                [['text' => '🔙 بازگشت به منو', 'callback_data' => 'back_to_menu']]
            ]
        ];
        
        $this->editMessageText($chat_id, $message_id, $text, $keyboard);
    }
    
    // پردازش پیام متنی
    private function handleTextMessage($chat_id, $user, $text) {
        if (isset($_SESSION['waiting_for_token'])) {
            $this->processBotToken($chat_id, $user, $text);
        } elseif (isset($_SESSION['waiting_for_consultation'])) {
            $this->processConsultation($chat_id, $user, $text);
        } else {
            $this->sendMessage($chat_id, "❓ دستور نامشخص! لطفاً از منو استفاده کنید.");
        }
    }
    
    // پردازش توکن ربات
    private function processBotToken($chat_id, $user, $token) {
        $bot_type = $_SESSION['waiting_for_token'];
        unset($_SESSION['waiting_for_token']);
        
        global $available_bots;
        $bot = $available_bots[$bot_type];
        
        // اعتبارسنجی توکن (ساده)
        if (strlen($token) < 20 || strpos($token, ':') === false) {
            $this->sendMessage($chat_id, "❌ توکن نامعتبر! لطفاً توکن صحیح را ارسال کنید.");
            return;
        }
        
        // ثبت سفارش
        if ($this->db->createOrder($user['id'], $bot_type, $token, $bot['price'])) {
            // کسر موجودی
            $this->db->updateBalance($user['id'], -$bot['price'], 'withdraw');
            
            // ارسال پیام به ادمین
            $this->notifyAdminNewOrder($user, $bot_type, $token, $bot['price']);
            
            $text = "✅ <b>سفارش شما ثبت شد!</b>\n\n";
            $text .= "🤖 ربات: " . $bot['name'] . "\n";
            $text .= "💰 مبلغ: " . number_format($bot['price']) . " " . CURRENCY . "\n";
            $text .= "📅 تاریخ: " . date('Y/m/d H:i') . "\n\n";
            $text .= "🔔 سفارش شما برای ادمین ارسال شد و به زودی پردازش خواهد شد.\n";
            $text .= "📞 در صورت نیاز با پشتیبانی تماس بگیرید.";
            
            $this->sendMessage($chat_id, $text);
        } else {
            $this->sendMessage($chat_id, "❌ خطا در ثبت سفارش! لطفاً دوباره تلاش کنید.");
        }
    }
    
    // پردازش درخواست مشاوره
    private function processConsultation($chat_id, $user, $text) {
        unset($_SESSION['waiting_for_consultation']);
        
        // ثبت درخواست مشاوره
        if ($this->db->createConsultation($user['id'], 'مشاوره', $text, $user['telegram_id'])) {
            // ارسال پیام به ادمین
            $this->notifyAdminNewConsultation($user, $text);
            
            $response = "✅ <b>درخواست مشاوره شما ثبت شد!</b>\n\n";
            $response .= "📝 موضوع: مشاوره\n";
            $response .= "📅 تاریخ: " . date('Y/m/d H:i') . "\n\n";
            $response .= "🔔 درخواست شما برای ادمین ارسال شد و به زودی پاسخ داده خواهد شد.\n";
            $response .= "📞 در صورت فوری بودن، با پشتیبانی تماس بگیرید.";
            
            $this->sendMessage($chat_id, $response);
        } else {
            $this->sendMessage($chat_id, "❌ خطا در ثبت درخواست! لطفاً دوباره تلاش کنید.");
        }
    }
    
    // اطلاع‌رسانی سفارش جدید به ادمین
    private function notifyAdminNewOrder($user, $bot_type, $token, $amount) {
        global $available_bots;
        $bot = $available_bots[$bot_type];
        
        $text = "🆕 <b>سفارش جدید</b>\n\n";
        $text .= "👤 کاربر: " . ($user['first_name'] ?? 'نامشخص') . " (@" . ($user['username'] ?? 'بدون نام کاربری') . ")\n";
        $text .= "🆔 شناسه: " . $user['telegram_id'] . "\n";
        $text .= "🤖 ربات: " . $bot['name'] . "\n";
        $text .= "💰 مبلغ: " . number_format($amount) . " " . CURRENCY . "\n";
        $text .= "🔑 توکن: " . substr($token, 0, 10) . "...\n";
        $text .= "📅 تاریخ: " . date('Y/m/d H:i') . "\n\n";
        $text .= "🔗 <a href='" . SITE_URL . "/admin.php'>مدیریت سفارشات</a>";
        
        $this->sendMessage(ADMIN_ID, $text);
    }
    
    // اطلاع‌رسانی درخواست مشاوره جدید به ادمین
    private function notifyAdminNewConsultation($user, $description) {
        $text = "📝 <b>درخواست مشاوره جدید</b>\n\n";
        $text .= "👤 کاربر: " . ($user['first_name'] ?? 'نامشخص') . " (@" . ($user['username'] ?? 'بدون نام کاربری') . ")\n";
        $text .= "🆔 شناسه: " . $user['telegram_id'] . "\n";
        $text .= "📝 توضیحات:\n" . $description . "\n";
        $text .= "📅 تاریخ: " . date('Y/m/d H:i') . "\n\n";
        $text .= "🔗 <a href='" . SITE_URL . "/admin.php'>مدیریت درخواست‌ها</a>";
        
        $this->sendMessage(ADMIN_ID, $text);
    }
}
?>
